/*
 * Routines for sending invalid routes
 */
#include <sys/types.h>
#include <netinet/in.h>
#include <time.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_fms_comm.h"
#include "lf_fms.h"

#include "fma.h"
#include "fma_fms.h"
#include "fma_fabric.h"
#include "fma_myri.h"
#include "fma_resolve.h"

/*
 * private struct for invalid route raw send callback
 */
struct fma_inv_rt_rslts {
  int send_succeeded;
  int seq_no;
  int xbar_index;
  int host_in_port;
};


/*
 * Local prototypes
 */
static void u_msleep(unsigned int msec);
static void dump_xbar_rts();
static void fma_send_inv_rts_complete(void *inv_rt_rslts);

static void
u_msleep(
  unsigned int msec)
{
  struct timespec t;

  t.tv_sec = msec / 1000;
  t.tv_nsec = (msec % 1000) * 1000000;
  (void) nanosleep(&t, NULL);
}

static
void dump_xbar_rts()
{
  int i;
  struct lf_fabric *fp;
  struct lf_xbar *x;

  fp = A.fabric;

  for (i = 0; i < fp->num_xbars; i++) {
    x = fp->xbars[i];
    if (FMA_XBAR(x)->host_dist == -1) {
      fma_log("xbar:%d, NO ROUTE", i);
    } else {
      fma_log("xbar:%d, from:%d,%d rt_len:%d %s", i,
        FMA_XBAR(x)->host_nic->nic_id,
        FMA_XBAR(x)->host_nic_port, FMA_XBAR(x)->host_dist,
	lf_route_str(FMA_XBAR(x)->host_route, FMA_XBAR(x)->host_dist));
    }
  }
}

static int onetime = 1;

int
fma_send_inv_rts(
  struct lf_fma_send_inv_rt *inv_rt)
{
  int i;
  int rc;
  int seq_no;
  int xbar_index;
  struct lf_fabric *fp;
  struct lf_xbar *x;
  int my_nic_id;
  int my_nic_ifc;
  unsigned char *route; 
  int route_len; 
  unsigned char pkt_data;
  struct fma_inv_rt_rslts *inv_rt_rslts;
  int num_pkts;

  fp = A.fabric;

  if (A.debug && onetime) {
    dump_xbar_rts();
    onetime = 0;
  }

  seq_no = ntohl(inv_rt->ir_seq_no_32);
  xbar_index = ntohl(inv_rt->ir_xbar_index_32);
  num_pkts = ntohl(inv_rt->ir_num_pkts_32);

  if (xbar_index >= fp->num_xbars) LF_ERROR(("Bad xbar index"));

  /* place for results */
  x = fp->xbars[xbar_index];
  LF_CALLOC(inv_rt_rslts, struct fma_inv_rt_rslts, 1);
  inv_rt_rslts->seq_no = seq_no;
  inv_rt_rslts->xbar_index = xbar_index;
  inv_rt_rslts->host_in_port = FMA_XBAR(x)->host_in_port;
  inv_rt_rslts->send_succeeded = TRUE;

  /* If we cannot reach the xbar, compplain about it to FMS */
  if (FMA_XBAR(x)->host_nic == NULL) {
    fma_log("Cannot reach requested xbar for invalid routes");
    inv_rt_rslts->send_succeeded = FALSE;
    fma_send_inv_rts_complete(inv_rt_rslts);
    return 0;
  }

  my_nic_id = FMA_XBAR(x)->host_nic->nic_id;
  my_nic_ifc = FMA_XBAR(x)->host_nic_port;
  route = FMA_XBAR(x)->host_route;
  route_len = FMA_XBAR(x)->host_dist;
  pkt_data = 0;

  if (A.debug) {
    fma_log(
      "sending invalid rt from %d:%d, len:%d, host_in_port:%d to xbar %d: %s",
        FMA_XBAR(x)->host_nic->nic_id, FMA_XBAR(x)->host_nic_port,
        FMA_XBAR(x)->host_dist, FMA_XBAR(x)->host_in_port, xbar_index,
    lf_route_str(FMA_XBAR(x)->host_route, FMA_XBAR(x)->host_dist));
  }

  for (i=0; i < num_pkts; i++) {
    rc = fma_myri_raw_send(my_nic_id, my_nic_ifc, route, route_len,
                     &pkt_data, 1, NULL, NULL);

    if (rc != 0) {
      inv_rt_rslts->send_succeeded = FALSE;
      fma_send_inv_rts_complete(inv_rt_rslts);
    }
    u_msleep(1);
  }
  rc = fma_myri_raw_send(my_nic_id, my_nic_ifc, route, route_len,
                     &pkt_data, 1, fma_send_inv_rts_complete, inv_rt_rslts);
  if (rc != 0) {
    inv_rt_rslts->send_succeeded = FALSE;
    fma_send_inv_rts_complete(inv_rt_rslts);
  }

  return rc;
 except:
  return -1;
}

/*
 * Send the results of the invald route sends back to FMS.
 */
static
void
fma_send_inv_rts_complete(
  void *inv_rt_msg_p)

{
  int rc;
  int sock;
  struct fma_inv_rt_rslts *inv_rt_rslts;
  struct fma_fms_inv_rt_msg msg;

  inv_rt_rslts = inv_rt_msg_p;

  sock = A.fms->socket;

  msg.irr_send_succeeded_32 = htonl(inv_rt_rslts->send_succeeded);
  msg.irr_seq_no_32 = htonl(inv_rt_rslts->seq_no);
  msg.irr_xbar_index_32 = htonl(inv_rt_rslts->xbar_index);
  msg.irr_host_in_port_32 = htonl(inv_rt_rslts->host_in_port);

  if (A.debug) {
    fma_log("fma_send_inv_rts_complete xbar node id: %d, rslt:%d, hp:%d",
        inv_rt_rslts->xbar_index, inv_rt_rslts->send_succeeded,
        inv_rt_rslts->host_in_port);
  }

  LF_FREE(inv_rt_rslts);  /* allocated in fma_send_inv_rts */

  /* send the header and the message */
  rc = lf_send_fms_header(sock,FMA_FMS_SEND_INVALID_ROUTE_COMPLETE,sizeof(msg));
  if (rc != 0) goto error;
  rc = lf_write(sock, &msg, sizeof(msg));
  if (rc != sizeof(msg)) goto error;

  return;

 error:
  LF_ERROR(("Error sending int_rt completion to FMS"));
 except:
  fma_perror();
}
